/*******************************************************************************
  ProcessChecker Demo
  Example submitted by David Clegg

  This program primarily demonstrates the use of the System.Diagnostics.Process
  class to control and monitor processes. It provides the ability to specify a
  list of  processes to monitor, and will restart them if they are not running,
  or not responding. Each monitored process can also have a list of dependant
  processes associated with it, and these dependant processes will be terminated
  if the monitored process is to be restarted.

  As a side effect, examples of the following techniques & classes can also be
  found:
  -	Object serialization and deserialization to a data file using the
	  System.Runtime.Serialization.Formatters.Binary.BinaryFormatter and
	  System.IO.FileStream classes (ProcessCheckerSettings.cs).
  - Raising standard and custom events (ProcessChecker.cs,
	  ProcessCheckerSettings.cs).
  - Reading file version information using the System.IO.FileInfo class
	  (AboutForm.cs)
  - Implementing a strongly typed collection class that implements the
	  System.Collections.CollectionBase class (ProcessClasses.cs).
  - Various controls in System.Windows.Forms.

*******************************************************************************/
using System;
using System.Drawing;
using System.Collections;
using System.ComponentModel;
using System.Windows.Forms;
using System.Data;
using System.Diagnostics;

namespace ProcessChecker
{
	/// <summary>
	/// Summary description for WinForm1.
	/// </summary>
	public class MainForm : System.Windows.Forms.Form
	{
		/// <summary>
		/// Required designer variable.
		/// </summary>
		private System.ComponentModel.IContainer components = null;
		private System.Windows.Forms.Panel panel1;
		private System.Windows.Forms.CheckBox cbEnabled;
		private System.Windows.Forms.Button btnClear;
		private System.Windows.Forms.Button btnClose;
		private System.Windows.Forms.ListView lvMain;
		private System.Windows.Forms.ColumnHeader colTime;
		private System.Windows.Forms.ColumnHeader colDetails;
		private System.Windows.Forms.MainMenu mmMain;
		private System.Windows.Forms.MenuItem menuItem1;
		private System.Windows.Forms.MenuItem miClose;
		private System.Windows.Forms.MenuItem menuItem3;
		private System.Windows.Forms.MenuItem miSetup;
		private System.Windows.Forms.MenuItem menuItem5;
		private System.Windows.Forms.MenuItem miAbout;

		private ProcessChecker processChecker;

		public MainForm()
		{
			//
			// Required for Windows Form Designer support
			//
			InitializeComponent();

			//
			// TODO: Add any constructor code after InitializeComponent call
			//
		}

		/// <summary>
		/// Clean up any resources being used.
		/// </summary>
		protected override void Dispose (bool disposing)
		{
			if (disposing)
			{
				if (components != null)
				{
					components.Dispose();
				}
			}
			base.Dispose(disposing);
		}

		#region Windows Form Designer generated code
		/// <summary>
		/// Required method for Designer support - do not modify
		/// the contents of this method with the code editor.
		/// </summary>
		private void InitializeComponent() {
			System.Resources.ResourceManager resources = new System.Resources.ResourceManager(typeof(MainForm));
			this.panel1 = new System.Windows.Forms.Panel();
			this.btnClose = new System.Windows.Forms.Button();
			this.btnClear = new System.Windows.Forms.Button();
			this.cbEnabled = new System.Windows.Forms.CheckBox();
			this.lvMain = new System.Windows.Forms.ListView();
			this.colTime = new System.Windows.Forms.ColumnHeader();
			this.colDetails = new System.Windows.Forms.ColumnHeader();
			this.mmMain = new System.Windows.Forms.MainMenu();
			this.menuItem1 = new System.Windows.Forms.MenuItem();
			this.miSetup = new System.Windows.Forms.MenuItem();
			this.menuItem3 = new System.Windows.Forms.MenuItem();
			this.miClose = new System.Windows.Forms.MenuItem();
			this.menuItem5 = new System.Windows.Forms.MenuItem();
			this.miAbout = new System.Windows.Forms.MenuItem();
			this.panel1.SuspendLayout();
			this.SuspendLayout();
			// 
			// panel1
			// 
			this.panel1.Controls.Add(this.btnClose);
			this.panel1.Controls.Add(this.btnClear);
			this.panel1.Controls.Add(this.cbEnabled);
			this.panel1.Dock = System.Windows.Forms.DockStyle.Bottom;
			this.panel1.Location = new System.Drawing.Point(0, 170);
			this.panel1.Name = "panel1";
			this.panel1.Size = new System.Drawing.Size(712, 40);
			this.panel1.TabIndex = 0;
			// 
			// btnClose
			// 
			this.btnClose.Anchor = ((System.Windows.Forms.AnchorStyles)((System.Windows.Forms.AnchorStyles.Bottom | System.Windows.Forms.AnchorStyles.Right)));
			this.btnClose.Location = new System.Drawing.Point(624, 8);
			this.btnClose.Name = "btnClose";
			this.btnClose.TabIndex = 2;
			this.btnClose.Text = "Close";
			this.btnClose.Click += new System.EventHandler(this.btnClose_Click);
			// 
			// btnClear
			// 
			this.btnClear.Location = new System.Drawing.Point(104, 8);
			this.btnClear.Name = "btnClear";
			this.btnClear.TabIndex = 1;
			this.btnClear.Text = "Clear";
			this.btnClear.Click += new System.EventHandler(this.btnClear_Click);
			// 
			// cbEnabled
			// 
			this.cbEnabled.Location = new System.Drawing.Point(16, 8);
			this.cbEnabled.Name = "cbEnabled";
			this.cbEnabled.Size = new System.Drawing.Size(72, 24);
			this.cbEnabled.TabIndex = 0;
			this.cbEnabled.Text = "Enabled";
			this.cbEnabled.Click += new System.EventHandler(this.cbEnabled_Click);
			this.cbEnabled.CheckedChanged += new System.EventHandler(this.cbEnabled_CheckedChanged);
			// 
			// lvMain
			// 
			this.lvMain.Columns.AddRange(new System.Windows.Forms.ColumnHeader[] {
						this.colTime,
						this.colDetails});
			this.lvMain.Dock = System.Windows.Forms.DockStyle.Fill;
			this.lvMain.FullRowSelect = true;
			this.lvMain.Location = new System.Drawing.Point(0, 0);
			this.lvMain.Name = "lvMain";
			this.lvMain.Size = new System.Drawing.Size(712, 170);
			this.lvMain.TabIndex = 1;
			this.lvMain.View = System.Windows.Forms.View.Details;
			// 
			// colTime
			// 
			this.colTime.Text = "Event Time";
			this.colTime.Width = 150;
			// 
			// colDetails
			// 
			this.colDetails.Text = "Details";
			this.colDetails.Width = 500;
			// 
			// mmMain
			// 
			this.mmMain.MenuItems.AddRange(new System.Windows.Forms.MenuItem[] {
						this.menuItem1,
						this.menuItem5});
			// 
			// menuItem1
			// 
			this.menuItem1.Index = 0;
			this.menuItem1.MenuItems.AddRange(new System.Windows.Forms.MenuItem[] {
						this.miSetup,
						this.menuItem3,
						this.miClose});
			this.menuItem1.Text = "File";
			// 
			// miSetup
			// 
			this.miSetup.Index = 0;
			this.miSetup.Text = "Setup...";
			this.miSetup.Click += new System.EventHandler(this.miSetup_Click);
			// 
			// menuItem3
			// 
			this.menuItem3.Index = 1;
			this.menuItem3.Text = "-";
			// 
			// miClose
			// 
			this.miClose.Index = 2;
			this.miClose.Text = "Close";
			this.miClose.Click += new System.EventHandler(this.btnClose_Click);
			// 
			// menuItem5
			// 
			this.menuItem5.Index = 1;
			this.menuItem5.MenuItems.AddRange(new System.Windows.Forms.MenuItem[] {
						this.miAbout});
			this.menuItem5.Text = "Help";
			// 
			// miAbout
			// 
			this.miAbout.Index = 0;
			this.miAbout.Text = "About...";
			this.miAbout.Click += new System.EventHandler(this.miAbout_Click);
			// 
			// MainForm
			// 
			this.AutoScaleBaseSize = new System.Drawing.Size(5, 13);
			this.ClientSize = new System.Drawing.Size(712, 210);
			this.Controls.Add(this.lvMain);
			this.Controls.Add(this.panel1);
			this.Icon = ((System.Drawing.Icon)(resources.GetObject("$this.Icon")));
			this.Menu = this.mmMain;
			this.Name = "MainForm";
			this.StartPosition = System.Windows.Forms.FormStartPosition.CenterScreen;
			this.Text = "Process Checker";
			this.Load += new System.EventHandler(this.MainForm_Load);
			this.panel1.ResumeLayout(false);
			this.ResumeLayout(false);
		}
		#endregion

		/// <summary>
		/// The main entry point for the application.
		/// </summary>
		[STAThread]
		static void Main()
		{
			Application.Run(new MainForm());
		}

		private void btnClose_Click(object sender, System.EventArgs e) {
			this.Close();
		}

		/// <summary>
		/// Create the ProcessChecker instance
		/// </summary>
		private void MainForm_Load(object sender, System.EventArgs e) {
			AddEvent("Process Checker started");

			//Attempt to load the settings
			ProcessCheckerSettings settings;
			try {
				settings = ProcessCheckerSettings.Load();
			}
			catch (Exception exception) {
				MessageBox.Show(string.Format("There was an error loading the settings from disk:\n\n {0}" +
											  "\n\nThe settings have been reverted to their default values.",
											  exception.Message));
				settings = new ProcessCheckerSettings();
			}

			//Create the ProcessChecker instance
			processChecker = new ProcessChecker(settings);
			processChecker.ProcessCheckerEvent += new ProcessCheckerEventNotify(ProcessCheckerEvent);
			processChecker.Settings.SettingsSaved += new EventHandler(Settings_Saved);

			PopulateControls();
		}

		/// <summary>
		/// To ensure the cbEnabled CheckBox is always indicative of the
		/// ProcessCheckerSettings.Enabled property, we will check its property
		/// everytime the settings have been saved (e.g a problem was encountered
		/// setting the Timers interval in the ProcessChecker instance).
		/// </summary>
		private void Settings_Saved(object sender, EventArgs args) {
			AddEvent("Settings Saved");
			cbEnabled.Checked = processChecker.Settings.Enabled;
		}

		private void PopulateControls() {
			//Enable/Disable processing
			cbEnabled.Checked = processChecker.Settings.Enabled;
		}

		/// <summary>
		/// Event handler fired when the ProcessChecker instance has something
		/// to tell us.
		/// </summary>
		private void ProcessCheckerEvent(object sender, ProcessCheckerEventArgs args) {
			AddEvent(args.Message);
		}

		/// <summary>
		/// Add a detail line to the ListView
		/// </summary>
		private void AddEvent(string eventDetails) {
		  ListViewItem li = new ListViewItem(DateTime.Now.ToString());
		  li.SubItems.Add(eventDetails);
		  lvMain.Items.Add(li);
		}

		/// <summary>
		/// Set the ProcessCheckerSettings.Enabled property. This is done here
		/// rather than in CheckBox.Check, as that event fires whenever
		/// CheckBox.Checked changes. This event only fires if CheckBox.Checked
		/// changes as a result of the CheckBox being clicked by the user.
		/// </summary>
		private void cbEnabled_Click(object sender, System.EventArgs e) {
			processChecker.Settings.Enabled = cbEnabled.Checked;
			processChecker.Settings.Save();
		}

		private void btnClear_Click(object sender, System.EventArgs e) {
			lvMain.Items.Clear();
		}

		private void miSetup_Click(object sender, System.EventArgs e) {
			//Temporarily disable process checking while we are in the
			//setup screen.
			processChecker.Suspend();
			try {
				SetupForm.ShowSetupForm(processChecker.Settings);
			}
			finally {
				processChecker.Resume();
			}
		}

		private void miAbout_Click(object sender, System.EventArgs e) {
			AboutForm.ShowAboutForm();
		}

		/// <summary>
		/// Enable/Disable process checking.
		/// </summary>
		private void cbEnabled_CheckedChanged(object sender, System.EventArgs e) {
			if (cbEnabled.Checked) {
				AddEvent("Processing Enabled");
			}
			else {
				AddEvent("Processing Disabled");
			}
		}

	}
}
